<?php
/**
 * ===================================================================================
 * Manage Subjects Page (Mobile Responsive)
 * ===================================================================================
 *
 * File: manage_subjects.php
 * Location: /homework_portal/admin/manage_subjects.php
 *
 * Purpose:
 * 1. Provides a secure and responsive interface for Admins to manage school subjects.
 * 2. Access is controlled by the check_permission() function.
 * 3. Adapts for mobile, tablet, and desktop screens.
 */

require_once '../config.php';
require_once 'admin_helpers.php'; // Correctly include the helper file

// Security Check for this specific page
check_permission('manage_subjects');

$errors = [];
$success_message = '';

// Handle feedback messages from session
if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}
if (isset($_SESSION['error_message'])) {
    $errors[] = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
}


// --- Handle Form Submission to CREATE a new subject ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_subject'])) {
    $subject_name = trim($_POST['subject_name'] ?? '');

    if (empty($subject_name)) {
        $errors[] = "Subject name is required.";
    } else {
        $stmt = $pdo->prepare("SELECT id FROM subjects WHERE subject_name = ?");
        $stmt->execute([$subject_name]);
        if ($stmt->fetch()) {
            $errors[] = "This subject already exists.";
        }
    }

    if (empty($errors)) {
        $sql = "INSERT INTO subjects (subject_name) VALUES (?)";
        try {
            $pdo->prepare($sql)->execute([$subject_name]);
            $_SESSION['success_message'] = "Subject '{$subject_name}' was created successfully!";
        } catch (PDOException $e) {
            $_SESSION['error_message'] = "Database error: Could not create subject.";
        }
    } else {
        $_SESSION['error_message'] = implode('<br>', $errors);
    }
    header('Location: manage_subjects.php');
    exit();
}

// --- Fetch all existing subjects and other data for display ---
try {
    $subjects = $pdo->query("SELECT id, subject_name FROM subjects ORDER BY subject_name ASC")->fetchAll();
    $unread_count = $pdo->query("SELECT COUNT(id) FROM teacher_messages WHERE sender_type = 'teacher' AND is_read = 0")->fetchColumn();
} catch (PDOException $e) {
    $subjects = [];
    $unread_count = 0;
    $errors[] = "Could not fetch data from the database.";
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Subjects - Admin Portal</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&family=Baloo+2:wght@700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Poppins', sans-serif; background-color: #f0f9ff; }
        .font-brand { font-family: 'Baloo 2', cursive; }
        .sidebar-link { display: flex; align-items: center; padding: 0.75rem 1rem; border-radius: 0.5rem; transition: background-color 0.2s, color 0.2s; }
        .sidebar-link:hover, .sidebar-link.active { background-color: #e0f2fe; color: #0284c7; }
        .notification-badge { display: inline-flex; align-items: center; justify-content: center; width: 22px; height: 22px; border-radius: 9999px; background-color: #ef4444; color: white; font-size: 0.75rem; font-weight: bold; }

        /* Styles for Responsive Table */
        @media (max-width: 768px) {
            .responsive-table thead { display: none; }
            .responsive-table tr {
                display: block;
                margin-bottom: 1rem;
                border: 1px solid #ddd;
                border-radius: 0.5rem;
                padding: 1rem;
                box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            }
            .responsive-table td {
                display: flex;
                justify-content: space-between;
                align-items: center;
                text-align: right;
                padding: 0.5rem 0;
                border-bottom: 1px solid #eee;
            }
            .responsive-table td:last-child {
                border-bottom: none;
                justify-content: center; /* Center action buttons */
                padding-top: 1rem;
            }
            .responsive-table td::before {
                content: attr(data-label);
                font-weight: 600;
                text-align: left;
                padding-right: 1rem;
            }
        }
    </style>
</head>
<body class="flex flex-col min-h-screen">
    <div class="relative min-h-screen md:flex">
        <!-- Mobile menu button -->
        <div class="md:hidden flex justify-between items-center bg-white p-4 shadow-md">
             <a href="dashboard.php">
                <h1 class="text-3xl font-bold text-red-600 font-sans">TTTEDDDEL</h1>
            </a>
            <button id="mobile-menu-button" class="text-gray-500 hover:text-blue-600 focus:outline-none">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16m-7 6h7"></path></svg>
            </button>
        </div>

        <!-- Sidebar Navigation -->
        <aside id="sidebar" class="w-64 bg-white shadow-md absolute inset-y-0 left-0 transform -translate-x-full md:relative md:translate-x-0 transition-transform duration-300 ease-in-out z-30">
            <div class="p-4 border-b text-center">
                <a href="dashboard.php">
                    <h1 class="text-4xl font-bold text-red-600 tracking-widest font-sans">TTTEDDDEL</h1>
                    <p class="text-xs font-semibold text-blue-800 mt-1">Teacher-Student Interactive System</p>
                    <p class="text-xs text-blue-800">by Ele Infocomm Technologies</p>
                </a>
            </div>
            <nav class="p-4 space-y-2">
                <a href="dashboard.php" class="sidebar-link"><span>Dashboard</span></a>
                <a href="inbox.php" class="sidebar-link flex justify-between items-center">
                    <span>Inbox</span>
                    <?php if ($unread_count > 0): ?><span class="notification-badge"><?php echo $unread_count; ?></span><?php endif; ?>
                </a>
                <?php if (has_permission('manage_admins')): ?><a href="manage_admins.php" class="sidebar-link"><span>Manage Admins</span></a><?php endif; ?>
                <?php if (has_permission('manage_teachers')): ?><a href="manage_teachers.php" class="sidebar-link"><span>Manage Teachers</span></a><?php endif; ?>
                <?php if (has_permission('manage_classes')): ?><a href="manage_classes.php" class="sidebar-link"><span>Manage Classes</span></a><?php endif; ?>
                <?php if (has_permission('manage_subjects')): ?><a href="manage_subjects.php" class="sidebar-link active"><span>Manage Subjects</span></a><?php endif; ?>
                <?php if (has_permission('assign_subjects')): ?><a href="assign_subjects.php" class="sidebar-link"><span>Assign Subjects</span></a><?php endif; ?>
                <?php if (has_permission('send_message')): ?><a href="send_message.php" class="sidebar-link"><span>Send Message</span></a><?php endif; ?>
            </nav>
        </aside>

        <!-- Main Content Area -->
        <div class="flex-1 flex flex-col overflow-hidden">
            <header class="bg-white shadow-sm border-b hidden md:flex"><div class="flex items-center justify-end p-4 w-full"><span class="text-gray-600 mr-4">Welcome, <?php echo e($_SESSION['admin_username'] ?? 'Admin'); ?>!</span><a href="account_settings.php" class="p-2 text-gray-500 hover:text-blue-600" title="Account Settings"><svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" /><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" /></svg></a><a href="logout.php" class="ml-2 py-2 px-4 bg-blue-600 text-white font-semibold rounded-lg shadow-md hover:bg-blue-700">Logout</a></div></header>

            <main class="flex-1 overflow-x-hidden overflow-y-auto bg-gray-100 p-4 md:p-6">
                <h2 class="text-3xl font-semibold text-gray-800">Manage Subjects</h2>
                
                <?php if (!empty($errors)): ?><div class="p-4 my-4 text-sm text-red-800 bg-red-100 rounded-lg"><ul class="list-disc list-inside"><?php foreach ($errors as $error) echo "<li>" . $error . "</li>"; ?></ul></div><?php endif; ?>
                <?php if ($success_message): ?><div class="p-4 my-4 text-sm text-green-800 bg-green-100 rounded-lg"><?php echo $success_message; ?></div><?php endif; ?>

                <!-- Create Subject Form -->
                <div class="mt-6 bg-white p-6 rounded-lg shadow">
                    <h3 class="text-xl font-semibold text-gray-800 mb-4">Create New Subject</h3>
                    <form action="manage_subjects.php" method="POST" class="grid grid-cols-1 md:grid-cols-3 gap-6 items-end">
                        <div class="md:col-span-2">
                            <label for="subject_name" class="block text-sm font-medium text-gray-700">Subject Name</label>
                            <input type="text" name="subject_name" id="subject_name" placeholder="e.g., Mathematics, History" class="mt-1 block w-full p-2 border border-gray-300 rounded-md shadow-sm" required>
                        </div>
                        <div class="text-center md:text-left">
                            <button type="submit" name="create_subject" class="w-full md:w-auto px-6 py-2 bg-blue-600 text-white font-semibold rounded-lg shadow hover:bg-blue-700">Create Subject</button>
                        </div>
                    </form>
                </div>

                <!-- Subjects List -->
                <div class="mt-8 bg-white shadow-lg rounded-lg overflow-hidden">
                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200 responsive-table">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Subject Name</th>
                                    <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php if (empty($subjects)): ?>
                                    <tr><td colspan="2" class="text-center p-6 text-gray-500">No subjects found. Start by creating one above.</td></tr>
                                <?php else: ?>
                                    <?php foreach ($subjects as $subject): ?>
                                        <tr>
                                            <td data-label="Subject" class="px-6 py-4 text-sm font-medium text-gray-900"><?php echo e($subject['subject_name']); ?></td>
                                            <td data-label="Actions" class="px-6 py-4 whitespace-nowrap text-sm font-medium text-center space-x-4">
                                                <a href="edit_subject.php?id=<?php echo $subject['id']; ?>" class="text-indigo-600 hover:text-indigo-900">Edit</a>
                                                <a href="subject_actions.php?action=delete&id=<?php echo $subject['id']; ?>" onclick="return confirm('Are you sure you want to delete this subject?');" class="text-red-600 hover:text-red-900">Delete</a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <footer class="text-center py-6 bg-gray-100 border-t mt-auto">
        <p class="text-sm text-gray-500">Built By Ele InfoComm Technologies- Kenya | <a href="http://web.e-ict.co.ke" target="_blank" class="hover:underline">web.e-ict.co.ke</a> | <a href="mailto:info@e-ict.co.ke" class="hover:underline">info@e-ict.co.ke</a> | +254202000121 | +254 722 215810 | 2025 All right reserved.</p>
    </footer>

    <script>
        const btn = document.getElementById('mobile-menu-button');
        const sidebar = document.getElementById('sidebar');

        btn.addEventListener('click', () => {
            sidebar.classList.toggle('-translate-x-full');
        });
    </script>
</body>
</html>
